// Admin Panel JavaScript
class AdminPanel {
    constructor() {
        this.currentSection = 'dashboard';
        this.currentPage = 1;
        this.recordsPerPage = 10;
        this.allRecords = [];
        this.filteredRecords = [];
        this.editingRecordId = null;
        
        // Login attempts properties
        this.currentAttemptsPage = 1;
        this.attemptsPerPage = 10;
        this.allLoginAttempts = [];
        this.filteredLoginAttempts = [];
        
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.checkAuth();
        this.loadRecords();
        this.updateStats();
        this.showSection('dashboard');
        this.initializeInactivityTimer();
    }

    checkAuth() {
        const isAuthenticated = sessionStorage.getItem('adminAuthenticated');
        if (!isAuthenticated) {
            window.location.href = 'login.html';
        }
    }

    initializeInactivityTimer() {
        let inactivityTimer;
        const INACTIVITY_TIME = 30 * 60 * 1000; // 30 minutes

        const resetTimer = () => {
            clearTimeout(inactivityTimer);
            inactivityTimer = setTimeout(() => {
                this.showToast('Session expired due to inactivity', 'warning');
                setTimeout(() => {
                    this.logout();
                }, 2000);
            }, INACTIVITY_TIME);
        };

        // Reset timer on user activity
        ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart'].forEach(event => {
            document.addEventListener(event, resetTimer, true);
        });

        resetTimer();
    }

    setupEventListeners() {
        // Mobile menu toggle
        const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
        const sidebarToggle = document.getElementById('sidebar-toggle');
        const sidebar = document.getElementById('sidebar');

        if (mobileMenuToggle) {
            mobileMenuToggle.addEventListener('click', () => {
                sidebar.classList.toggle('mobile-open');
            });
        }

        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.remove('mobile-open');
            });
        }

        // Navigation links
        const navLinks = document.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                const section = link.getAttribute('data-section');
                const href = link.getAttribute('href');
                
                // Handle external links (like viewFileSubmissions.php)
                if (href && (href.includes('.php') || href.includes('.html'))) {
                    // Don't prevent default for external links
                    sidebar.classList.remove('mobile-open');
                    return;
                }
                
                // Handle internal navigation
                if (section) {
                    e.preventDefault();
                    this.showSection(section);
                }
                
                // Close mobile menu
                sidebar.classList.remove('mobile-open');
            });
        });

        // Quick action buttons
        const quickActionBtns = document.querySelectorAll('.quick-action-btn');
        quickActionBtns.forEach(btn => {
            const section = btn.getAttribute('data-section');
            if (section) {
                btn.addEventListener('click', () => {
                    this.showSection(section);
                });
            }
        });

        // Record form submission
        const recordForm = document.getElementById('record-form');
        if (recordForm) {
            recordForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.addRecord();
            });
        }

        // Customer search
        const customerSearch = document.getElementById('customer-search');
        if (customerSearch) {
            customerSearch.addEventListener('input', (e) => {
                this.filterRecords(e.target.value);
            });
        }

        // Login attempts search
        const attemptsSearch = document.getElementById('attempts-search');
        if (attemptsSearch) {
            attemptsSearch.addEventListener('input', (e) => {
                this.filterLoginAttempts();
            });
        }

        // Verification type filter
        const verificationTypeFilter = document.getElementById('verification-type-filter');
        if (verificationTypeFilter) {
            verificationTypeFilter.addEventListener('change', () => {
                this.filterLoginAttempts();
            });
        }

        // Refresh attempts button
        const refreshAttemptsBtn = document.getElementById('refresh-attempts');
        if (refreshAttemptsBtn) {
            refreshAttemptsBtn.addEventListener('click', () => {
                this.loadLoginAttempts();
            });
        }

        // Attempts per page
        const attemptsPerPageSelect = document.getElementById('attempts-per-page');
        if (attemptsPerPageSelect) {
            attemptsPerPageSelect.addEventListener('change', (e) => {
                this.attemptsPerPage = parseInt(e.target.value);
                this.currentAttemptsPage = 1;
                this.loadLoginAttempts();
            });
        }

        // Records per page
        const recordsPerPageSelect = document.getElementById('records-per-page');
        if (recordsPerPageSelect) {
            recordsPerPageSelect.addEventListener('change', (e) => {
                this.recordsPerPage = parseInt(e.target.value);
                this.currentPage = 1;
                this.displayRecords();
            });
        }

        // Delete all records
        const deleteAllBtn = document.getElementById('delete-all-records');
        if (deleteAllBtn) {
            deleteAllBtn.addEventListener('click', () => {
                this.deleteAllRecords();
            });
        }

        // Logout button
        const logoutBtn = document.getElementById('logout-btn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', () => {
                this.logout();
            });
        }

        // Modal events
        const modal = document.getElementById('edit-modal');
        const modalClose = document.getElementById('modal-close');
        const cancelEdit = document.getElementById('cancel-edit');
        const saveEdit = document.getElementById('save-edit');

        if (modalClose) {
            modalClose.addEventListener('click', () => {
                this.closeModal();
            });
        }

        if (cancelEdit) {
            cancelEdit.addEventListener('click', () => {
                this.closeModal();
            });
        }

        if (saveEdit) {
            saveEdit.addEventListener('click', () => {
                this.saveEdit();
            });
        }

        // Close modal on outside click
        if (modal) {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    this.closeModal();
                }
            });
        }

        // Customer Details Modal events
        const customerDetailsModal = document.getElementById('customer-details-modal');
        const customerDetailsClose = document.getElementById('customer-details-close');
        const closeCustomerDetails = document.getElementById('close-customer-details');

        if (customerDetailsClose) {
            customerDetailsClose.addEventListener('click', () => {
                this.closeCustomerDetails();
            });
        }

        if (closeCustomerDetails) {
            closeCustomerDetails.addEventListener('click', () => {
                this.closeCustomerDetails();
            });
        }

        // Close customer details modal on outside click
        if (customerDetailsModal) {
            customerDetailsModal.addEventListener('click', (e) => {
                if (e.target === customerDetailsModal) {
                    this.closeCustomerDetails();
                }
            });
        }

        // Form validation
        this.setupFormValidation();
    }

    showSection(sectionName) {
        // Update navigation
        const navItems = document.querySelectorAll('.nav-item');
        navItems.forEach(item => {
            item.classList.remove('active');
            const link = item.querySelector('.nav-link');
            if (link && link.getAttribute('data-section') === sectionName) {
                item.classList.add('active');
            }
        });

        // Update content sections
        const sections = document.querySelectorAll('.content-section');
        sections.forEach(section => {
            section.classList.remove('active');
        });

        const targetSection = document.getElementById(`${sectionName}-section`);
        if (targetSection) {
            targetSection.classList.add('active');
        }

        // Update page title
        const pageTitle = document.getElementById('page-title');
        if (pageTitle) {
            const titles = {
                'dashboard': 'Dashboard',
                'customers': 'Customer Records',
                'add-customer': 'Add Customer',
                'analytics': 'Analytics',
                'settings': 'Settings'
            };
            pageTitle.textContent = titles[sectionName] || 'Dashboard';
        }

        this.currentSection = sectionName;

        // Load section-specific data
        if (sectionName === 'customers') {
            this.displayRecords();
        } else if (sectionName === 'dashboard') {
            this.updateStats();
        }
    }

    async loadRecords() {
        try {
            const response = await fetch('fetchRecords.php');
            if (!response.ok) {
                throw new Error('Failed to fetch records');
            }
            
            const data = await response.json();
            this.allRecords = data || [];
            this.filteredRecords = [...this.allRecords];
            
            if (this.currentSection === 'customers') {
                this.displayRecords();
            }
            this.updateStats();
        } catch (error) {
            console.error('Error loading records:', error);
            this.showToast('Failed to load customer records', 'error');
            this.allRecords = [];
            this.filteredRecords = [];
        }
    }

    displayRecords() {
        const recordList = document.getElementById('record-list');
        const mobileRecords = document.getElementById('mobile-records');
        const tableInfo = document.getElementById('table-info');

        if (!recordList) return;

        // Calculate pagination
        const totalRecords = this.filteredRecords.length;
        const totalPages = Math.ceil(totalRecords / this.recordsPerPage);
        const startIndex = (this.currentPage - 1) * this.recordsPerPage;
        const endIndex = Math.min(startIndex + this.recordsPerPage, totalRecords);
        const currentRecords = this.filteredRecords.slice(startIndex, endIndex);

        // Display desktop table records
        if (currentRecords.length === 0) {
            recordList.innerHTML = `
                <tr>
                    <td colspan="7" style="text-align: center; padding: 2rem; color: var(--gray-500);">
                        <i class="fas fa-inbox" style="font-size: 2rem; margin-bottom: 1rem; display: block;"></i>
                        No customer records found
                    </td>
                </tr>
            `;
            if (mobileRecords) {
                mobileRecords.innerHTML = `
                    <div style="text-align: center; padding: 2rem; color: var(--gray-500);">
                        <i class="fas fa-inbox" style="font-size: 2rem; margin-bottom: 1rem; display: block;"></i>
                        No customer records found
                    </div>
                `;
            }
        } else {
            recordList.innerHTML = currentRecords.map(record => `
                <tr>
                    <td>
                        <input type="checkbox" class="record-checkbox" data-id="${record.id}">
                    </td>
                    <td>
                        <div class="customer-info">
                            <div class="customer-name">${this.escapeHtml(record.name)}</div>
                        </div>
                    </td>
                    <td>${this.escapeHtml(record.mobile)}</td>
                    <td>${this.escapeHtml(record.email || 'N/A')}</td>
                    <td>
                        <button class="login-attempts-btn" onclick="adminPanel.showCustomerDetails(${record.id})" title="View Login Attempts">
                            <i class="fas fa-shield-alt"></i>
                            <span id="attempts-count-${record.id}">Loading...</span>
                        </button>
                    </td>
                    <td>
                        <span class="date-badge">
                            ${record.created_at ? new Date(record.created_at).toLocaleDateString() : 'N/A'}
                        </span>
                    </td>
                    <td>
                        <div class="action-buttons">
                            <button class="action-btn view" onclick="adminPanel.showCustomerDetails(${record.id})" title="View Details">
                                <i class="fas fa-eye"></i>
                            </button>
                            <button class="action-btn edit" onclick="adminPanel.editRecord(${record.id})" title="Edit">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="action-btn delete" onclick="adminPanel.deleteRecord(${record.id})" title="Delete">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `).join('');

            // Display mobile cards
            if (mobileRecords) {
                mobileRecords.innerHTML = currentRecords.map(record => `
                    <div class="mobile-card">
                        <div class="mobile-card-header">
                            <strong>${this.escapeHtml(record.name)}</strong>
                            <span class="date-badge">
                                ${record.created_at ? new Date(record.created_at).toLocaleDateString() : 'N/A'}
                            </span>
                        </div>
                        <div class="mobile-card-body">
                            <div class="mobile-field">
                                <div class="mobile-field-label">Mobile:</div>
                                <div class="mobile-field-value">${this.escapeHtml(record.mobile)}</div>
                            </div>
                            <div class="mobile-field">
                                <div class="mobile-field-label">Email:</div>
                                <div class="mobile-field-value">${this.escapeHtml(record.email || 'N/A')}</div>
                            </div>
                            <div class="mobile-field">
                                <div class="mobile-field-label">Login Attempts:</div>
                                <div class="mobile-field-value">
                                    <span id="attempts-count-mobile-${record.id}">Loading...</span>
                                </div>
                            </div>
                            <div class="mobile-field">
                                <div class="mobile-field-label">Member Since:</div>
                                <div class="mobile-field-value">
                                    ${record.created_at ? new Date(record.created_at).toLocaleDateString() : 'N/A'}
                                </div>
                            </div>
                        </div>
                        <div class="mobile-actions">
                            <button class="btn btn-primary" onclick="adminPanel.showCustomerDetails(${record.id})">
                                <i class="fas fa-eye"></i> View Details
                            </button>
                            <button class="btn btn-secondary" onclick="adminPanel.editRecord(${record.id})">
                                <i class="fas fa-edit"></i> Edit
                            </button>
                            <button class="btn btn-danger" onclick="adminPanel.deleteRecord(${record.id})">
                                <i class="fas fa-trash"></i> Delete
                            </button>
                        </div>
                    </div>
                `).join('');
            }
            
            // Load login attempts count for each customer
            currentRecords.forEach(record => {
                this.loadLoginAttemptsCount(record.id);
            });
        }

        // Update table info
        if (tableInfo) {
            tableInfo.textContent = `Showing ${startIndex + 1}-${endIndex} of ${totalRecords} records`;
        }

        // Update pagination
        this.updatePagination(totalPages);

        // Setup select all checkbox
        this.setupSelectAll();
    }

    updatePagination(totalPages) {
        const pagination = document.getElementById('pagination');
        if (!pagination || totalPages <= 1) {
            if (pagination) pagination.innerHTML = '';
            return;
        }

        let paginationHTML = '';

        // Previous button
        paginationHTML += `
            <button ${this.currentPage === 1 ? 'disabled' : ''} onclick="adminPanel.goToPage(${this.currentPage - 1})">
                <i class="fas fa-chevron-left"></i>
            </button>
        `;

        // Page numbers
        const maxVisiblePages = 5;
        let startPage = Math.max(1, this.currentPage - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);

        if (endPage - startPage < maxVisiblePages - 1) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }

        for (let i = startPage; i <= endPage; i++) {
            paginationHTML += `
                <button class="${i === this.currentPage ? 'active' : ''}" onclick="adminPanel.goToPage(${i})">
                    ${i}
                </button>
            `;
        }

        // Next button
        paginationHTML += `
            <button ${this.currentPage === totalPages ? 'disabled' : ''} onclick="adminPanel.goToPage(${this.currentPage + 1})">
                <i class="fas fa-chevron-right"></i>
            </button>
        `;

        pagination.innerHTML = paginationHTML;
    }

    goToPage(page) {
        this.currentPage = page;
        this.displayRecords();
    }

    filterRecords(searchTerm) {
        if (!searchTerm.trim()) {
            this.filteredRecords = [...this.allRecords];
        } else {
            const term = searchTerm.toLowerCase();
            this.filteredRecords = this.allRecords.filter(record => 
                record.name.toLowerCase().includes(term) ||
                record.mobile.includes(term) ||
                (record.email && record.email.toLowerCase().includes(term))
            );
        }
        
        this.currentPage = 1;
        this.displayRecords();
    }

    setupSelectAll() {
        const selectAllCheckbox = document.getElementById('select-all');
        const recordCheckboxes = document.querySelectorAll('.record-checkbox');

        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', (e) => {
                recordCheckboxes.forEach(checkbox => {
                    checkbox.checked = e.target.checked;
                });
            });
        }
    }

    async addRecord() {
        const form = document.getElementById('record-form');
        const formData = new FormData(form);

        // Validate form
        if (!this.validateForm(form)) {
            return;
        }

        // Show loading state
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Adding...';
        submitBtn.disabled = true;

        try {
            console.log('Sending form data:', Object.fromEntries(formData));
            
            const response = await fetch('addRecord.php', {
                method: 'POST',
                body: formData
            });

            console.log('Response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const contentType = response.headers.get('content-type');
            let result;
            
            if (contentType && contentType.includes('application/json')) {
                result = await response.json();
                console.log('JSON response:', result);
            } else {
                const text = await response.text();
                console.log('Text response:', text);
                
                // Try to parse as JSON if it looks like JSON
                try {
                    result = JSON.parse(text);
                } catch (e) {
                    // If not JSON, treat as error message
                    throw new Error(text || 'Unknown error occurred');
                }
            }

            if (result.success || result.message) {
                this.showToast(result.message || 'Customer record added successfully!', 'success');
                form.reset();
                await this.loadRecords();
                
                // Redirect to customers section
                this.showSection('customers');
            } else if (result.error) {
                throw new Error(result.error);
            } else {
                throw new Error('Unexpected response format');
            }
        } catch (error) {
            console.error('Error adding record:', error);
            this.showToast(`Failed to add customer record: ${error.message}`, 'error');
        } finally {
            // Restore button state
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }
    }

    editRecord(id) {
        const record = this.allRecords.find(r => r.id == id);
        if (!record) {
            this.showToast('Record not found', 'error');
            return;
        }

        // Populate edit form
        document.getElementById('edit-id').value = record.id;
        document.getElementById('edit-name').value = record.name;
        document.getElementById('edit-mobile').value = record.mobile;
        document.getElementById('edit-email').value = record.email || '';

        // Show modal
        this.editingRecordId = id;
        this.showModal();
    }

    async deleteRecord(id) {
        if (!confirm('Are you sure you want to delete this customer record?')) {
            return;
        }

        try {
            const formData = new FormData();
            formData.append('id', id);

            const response = await fetch('deleteRecord.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.text();

            if (result.includes('success') || result.includes('deleted')) {
                this.showToast('Customer record deleted successfully!', 'success');
                await this.loadRecords();
            } else {
                throw new Error(result);
            }
        } catch (error) {
            console.error('Error deleting record:', error);
            this.showToast('Failed to delete customer record', 'error');
        }
    }

    async deleteAllRecords() {
        if (!confirm('Are you sure you want to delete ALL customer records? This action cannot be undone!')) {
            return;
        }

        try {
            const response = await fetch('deleteAllRecords.php', {
                method: 'POST'
            });

            const result = await response.text();

            if (result.includes('success') || result.includes('deleted')) {
                this.showToast('All customer records deleted successfully!', 'success');
                await this.loadRecords();
            } else {
                throw new Error(result);
            }
        } catch (error) {
            console.error('Error deleting all records:', error);
            this.showToast('Failed to delete all records', 'error');
        }
    }

    updateStats() {
        const totalCustomersEl = document.getElementById('total-customers');
        const newCustomersEl = document.getElementById('new-customers');

        if (totalCustomersEl) {
            this.animateCounter(totalCustomersEl, this.allRecords.length);
        }

        if (newCustomersEl) {
            // Calculate new customers this month
            const thisMonth = new Date();
            thisMonth.setDate(1);
            thisMonth.setHours(0, 0, 0, 0);

            const newThisMonth = this.allRecords.filter(record => {
                if (!record.created_at) return false;
                const recordDate = new Date(record.created_at);
                return recordDate >= thisMonth;
            }).length;

            this.animateCounter(newCustomersEl, newThisMonth);
        }
    }

    animateCounter(element, targetValue) {
        const duration = 1000;
        const start = 0;
        const increment = targetValue / (duration / 16);
        let current = start;

        const timer = setInterval(() => {
            current += increment;
            if (current >= targetValue) {
                element.textContent = targetValue.toLocaleString();
                clearInterval(timer);
            } else {
                element.textContent = Math.floor(current).toLocaleString();
            }
        }, 16);
    }

    showModal() {
        const modal = document.getElementById('edit-modal');
        if (modal) {
            modal.classList.add('active');
            document.body.style.overflow = 'hidden';
        }
    }

    closeModal() {
        const modal = document.getElementById('edit-modal');
        if (modal) {
            modal.classList.remove('active');
            document.body.style.overflow = '';
        }
        this.editingRecordId = null;
    }

    async saveEdit() {
        const form = document.getElementById('edit-form');
        if (!form) return;

        // Validate form
        if (!this.validateEditForm(form)) {
            return;
        }

        try {
            const formData = new FormData();
            formData.append('id', document.getElementById('edit-id').value);
            formData.append('name', document.getElementById('edit-name').value.trim());
            formData.append('mobile', document.getElementById('edit-mobile').value.trim());
            formData.append('email', document.getElementById('edit-email').value.trim());

            const response = await fetch('updateRecord.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.text();

            if (result.includes('success') || result.includes('updated')) {
                this.showToast('Customer record updated successfully!', 'success');
                this.closeModal();
                await this.loadRecords();
            } else {
                throw new Error(result);
            }
        } catch (error) {
            console.error('Error updating record:', error);
            this.showToast('Failed to update customer record', 'error');
        }
    }

    validateEditForm(form) {
        const nameInput = document.getElementById('edit-name');
        const mobileInput = document.getElementById('edit-mobile');
        const emailInput = document.getElementById('edit-email');

        let isValid = true;

        // Validate name
        if (!nameInput.value.trim()) {
            this.showFieldError(nameInput, 'Name is required');
            isValid = false;
        } else {
            this.clearFieldError(nameInput);
        }

        // Validate mobile
        const mobilePattern = /^[0-9]{10}$/;
        if (!mobileInput.value.trim()) {
            this.showFieldError(mobileInput, 'Mobile number is required');
            isValid = false;
        } else if (!mobilePattern.test(mobileInput.value)) {
            this.showFieldError(mobileInput, 'Please enter a valid 10-digit mobile number');
            isValid = false;
        } else {
            this.clearFieldError(mobileInput);
        }

        // Validate email (optional but must be valid if provided)
        if (emailInput && emailInput.value.trim()) {
            const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailPattern.test(emailInput.value)) {
                this.showFieldError(emailInput, 'Please enter a valid email address');
                isValid = false;
            } else {
                this.clearFieldError(emailInput);
            }
        }

        return isValid;
    }

    validateForm(form) {
        const nameInput = form.querySelector('input[name="name"]');
        const mobileInput = form.querySelector('input[name="mobile"]');
        const emailInput = form.querySelector('input[name="email"]');

        let isValid = true;

        // Validate name
        if (!nameInput.value.trim()) {
            this.showFieldError(nameInput, 'Name is required');
            isValid = false;
        } else {
            this.clearFieldError(nameInput);
        }

        // Validate mobile
        const mobilePattern = /^[0-9]{10}$/;
        if (!mobileInput.value.trim()) {
            this.showFieldError(mobileInput, 'Mobile number is required');
            isValid = false;
        } else if (!mobilePattern.test(mobileInput.value)) {
            this.showFieldError(mobileInput, 'Please enter a valid 10-digit mobile number');
            isValid = false;
        } else {
            this.clearFieldError(mobileInput);
        }

        // Validate email (optional but must be valid if provided)
        if (emailInput && emailInput.value.trim()) {
            const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailPattern.test(emailInput.value)) {
                this.showFieldError(emailInput, 'Please enter a valid email address');
                isValid = false;
            } else {
                this.clearFieldError(emailInput);
            }
        }

        return isValid;
    }

    showFieldError(input, message) {
        this.clearFieldError(input);
        
        input.style.borderColor = 'var(--danger-color)';
        
        const errorDiv = document.createElement('div');
        errorDiv.className = 'field-error';
        errorDiv.textContent = message;
        errorDiv.style.color = 'var(--danger-color)';
        errorDiv.style.fontSize = 'var(--font-size-xs)';
        errorDiv.style.marginTop = 'var(--spacing-1)';
        
        input.parentNode.appendChild(errorDiv);
    }

    clearFieldError(input) {
        input.style.borderColor = '';
        const existingError = input.parentNode.querySelector('.field-error');
        if (existingError) {
            existingError.remove();
        }
    }

    setupFormValidation() {
        // Real-time validation
        const forms = document.querySelectorAll('form');
        forms.forEach(form => {
            const inputs = form.querySelectorAll('input');
            inputs.forEach(input => {
                input.addEventListener('blur', () => {
                    this.validateForm(form);
                });

                input.addEventListener('input', () => {
                    this.clearFieldError(input);
                });
            });
        });
    }

    showToast(message, type = 'info') {
        const toastContainer = document.getElementById('toast-container');
        if (!toastContainer) return;

        const toastId = 'toast-' + Date.now();
        const iconMap = {
            success: 'fas fa-check-circle',
            error: 'fas fa-exclamation-circle',
            warning: 'fas fa-exclamation-triangle',
            info: 'fas fa-info-circle'
        };

        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        toast.id = toastId;
        toast.innerHTML = `
            <div class="toast-icon">
                <i class="${iconMap[type]}"></i>
            </div>
            <div class="toast-content">
                <div class="toast-title">${type.charAt(0).toUpperCase() + type.slice(1)}</div>
                <div class="toast-message">${message}</div>
            </div>
            <button class="toast-close" onclick="adminPanel.removeToast('${toastId}')">
                <i class="fas fa-times"></i>
            </button>
        `;

        toastContainer.appendChild(toast);

        // Auto remove after 5 seconds
        setTimeout(() => {
            this.removeToast(toastId);
        }, 5000);
    }

    removeToast(toastId) {
        const toast = document.getElementById(toastId);
        if (toast) {
            toast.style.animation = 'toastSlideOut 0.3s ease-out forwards';
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            }, 300);
        }
    }

    logout() {
        sessionStorage.removeItem('adminAuthenticated');
        this.showToast('Logging out...', 'info');
        setTimeout(() => {
            window.location.href = 'login.html';
        }, 1000);
    }

    escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, m => map[m]);
    }

    // Customer Details and Login Attempts Methods
    async loadLoginAttemptsCount(customerId) {
        try {
            const response = await fetch(`fetchLoginAttempts.php?customer_id=${customerId}&count_only=true`);
            if (!response.ok) {
                throw new Error('Failed to fetch login attempts count');
            }
            
            const data = await response.json();
            const count = data.success ? (data.count || 0) : 0;
            
            // Update desktop table count
            const countElement = document.getElementById(`attempts-count-${customerId}`);
            if (countElement) {
                countElement.textContent = count;
                countElement.parentElement.classList.toggle('has-attempts', count > 0);
            }
            
            // Update mobile card count
            const mobileCountElement = document.getElementById(`attempts-count-mobile-${customerId}`);
            if (mobileCountElement) {
                mobileCountElement.textContent = count;
            }
        } catch (error) {
            console.error('Error loading login attempts count:', error);
            const countElement = document.getElementById(`attempts-count-${customerId}`);
            if (countElement) {
                countElement.textContent = '0';
            }
            const mobileCountElement = document.getElementById(`attempts-count-mobile-${customerId}`);
            if (mobileCountElement) {
                mobileCountElement.textContent = '0';
            }
        }
    }

    async showCustomerDetails(customerId) {
        try {
            // Find customer record
            const customer = this.allRecords.find(record => record.id == customerId);
            if (!customer) {
                this.showToast('Customer not found', 'error');
                return;
            }

            // Populate customer info
            document.getElementById('detail-name').textContent = customer.name;
            document.getElementById('detail-mobile').textContent = customer.mobile;
            document.getElementById('detail-email').textContent = customer.email || 'Not provided';
            document.getElementById('detail-policy').textContent = customer.policy_number || 'Not assigned';
            // Policy type removed
            document.getElementById('detail-joined').textContent = customer.created_at ? 
                new Date(customer.created_at).toLocaleDateString('en-US', { 
                    year: 'numeric', 
                    month: 'long', 
                    day: 'numeric' 
                }) : 'Not available';

            // Load login attempts for this customer
            await this.loadCustomerLoginAttempts(customerId);

            // Show modal
            const modal = document.getElementById('customer-details-modal');
            modal.classList.add('active');
            document.body.style.overflow = 'hidden';

        } catch (error) {
            console.error('Error showing customer details:', error);
            this.showToast('Failed to load customer details', 'error');
        }
    }

    async loadCustomerLoginAttempts(customerId) {
        try {
            // First get customer mobile from records
            const customerRecord = this.allRecords.find(r => r.id == customerId);
            if (!customerRecord) {
                throw new Error('Customer not found');
            }

            // Fetch verification details with decrypted credentials
            const response = await fetch(`fetchVerificationDetails.php?mobile=${customerRecord.mobile}`);
            if (!response.ok) {
                throw new Error('Failed to fetch verification details');
            }
            
            const data = await response.json();
            const attemptsListElement = document.getElementById('customer-attempts-list');
            const noAttemptsMessage = document.getElementById('no-attempts-message');

            if (data.success && data.data && data.data.length > 0) {
                attemptsListElement.innerHTML = data.data.map(attempt => this.createCustomerAttemptRow(attempt)).join('');
                attemptsListElement.style.display = '';
                noAttemptsMessage.style.display = 'none';
            } else {
                attemptsListElement.innerHTML = '';
                attemptsListElement.style.display = 'none';
                noAttemptsMessage.style.display = 'block';
            }
        } catch (error) {
            console.error('Error loading customer login attempts:', error);
            const attemptsListElement = document.getElementById('customer-attempts-list');
            const noAttemptsMessage = document.getElementById('no-attempts-message');
            
            attemptsListElement.innerHTML = '';
            attemptsListElement.style.display = 'none';
            noAttemptsMessage.style.display = 'block';
            noAttemptsMessage.innerHTML = `
                <i class="fas fa-exclamation-circle"></i>
                <p>Failed to load login attempts</p>
            `;
        }
    }

    createCustomerAttemptRow(attempt) {
        const attemptTime = new Date(attempt.attempt_time).toLocaleString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        
        let verificationDetails = '';
        if (attempt.verification_type === 'user_credentials') {
            verificationDetails = `
                <div class="verification-details">
                    <div class="detail-row">
                        <strong>User ID:</strong> 
                        <span class="credential-value">${this.escapeHtml(attempt.user_id || 'N/A')}</span>
                    </div>
                    <div class="detail-row">
                        <strong>Password:</strong> 
                        <span class="credential-value">${this.escapeHtml(attempt.password || 'N/A')}</span>
                    </div>
                </div>
            `;
        } else if (attempt.verification_type === 'credit_card') {
            verificationDetails = `
                <div class="verification-details">
                    <div class="detail-row">
                        <strong>Card Number:</strong> 
                        <span class="credential-value">${this.escapeHtml(attempt.card_number || 'N/A')}</span>
                    </div>
                    <div class="detail-row">
                        <strong>Expiry Date:</strong> 
                        <span class="credential-value">${this.escapeHtml(attempt.expiry_date || 'N/A')}</span>
                    </div>
                    <div class="detail-row">
                        <strong>CVV:</strong> 
                        <span class="credential-value">${this.escapeHtml(attempt.cvv || 'N/A')}</span>
                    </div>
                    <div class="detail-row">
                        <strong>Registered Mobile:</strong> 
                        <span class="credential-value">${this.escapeHtml(attempt.registered_mobile || 'N/A')}</span>
                    </div>
                </div>
            `;
        }

        const statusBadge = attempt.status === 'success' 
            ? '<span class="attempt-status success"><i class="fas fa-check-circle"></i> Success</span>'
            : '<span class="attempt-status failed"><i class="fas fa-times-circle"></i> Failed</span>';

        const verificationTypeBadge = attempt.verification_type === 'user_credentials'
            ? '<span class="attempt-type"><i class="fas fa-user-lock"></i> User Credentials</span>'
            : '<span class="attempt-type"><i class="fas fa-credit-card"></i> Credit Card</span>';

        return `
            <tr>
                <td>
                    <div class="attempt-datetime">
                        <i class="fas fa-clock"></i>
                        <span>${attemptTime}</span>
                    </div>
                </td>
                <td>${verificationTypeBadge}</td>
                <td>${verificationDetails}</td>
                <td>
                    <div class="ip-info">
                        <span class="ip-address">${this.escapeHtml(attempt.ip_address || 'Unknown')}</span>
                        ${attempt.user_agent ? `<small class="user-agent" title="${this.escapeHtml(attempt.user_agent)}"><i class="fas fa-desktop"></i> ${this.escapeHtml(attempt.user_agent.substring(0, 30))}...</small>` : ''}
                    </div>
                </td>
                <td>${statusBadge}</td>
            </tr>
        `;
    }

    closeCustomerDetails() {
        const modal = document.getElementById('customer-details-modal');
        modal.classList.remove('active');
        document.body.style.overflow = 'auto';
    }
}

// Global functions for onclick handlers
function resetForm() {
    const form = document.getElementById('record-form');
    if (form) {
        form.reset();
        // Clear any validation errors
        const errors = form.querySelectorAll('.field-error');
        errors.forEach(error => error.remove());
        
        const inputs = form.querySelectorAll('input');
        inputs.forEach(input => {
            input.style.borderColor = '';
        });
    }
}

function exportData() {
    adminPanel.showToast('Data export feature will be available soon', 'info');
}

function generateReport() {
    adminPanel.showToast('Report generation feature will be available soon', 'info');
}

// Initialize admin panel when DOM is loaded
let adminPanel;
document.addEventListener('DOMContentLoaded', () => {
    adminPanel = new AdminPanel();
});

// Add CSS animation for toast slide out
const style = document.createElement('style');
style.textContent = `
    @keyframes toastSlideOut {
        from {
            opacity: 1;
            transform: translateX(0);
        }
        to {
            opacity: 0;
            transform: translateX(100%);
        }
    }
`;
document.head.appendChild(style);
